<?php
namespace artelligence_Toolkit_Addon;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Plugin class.
 *
 * The main class that initiates and runs the addon.
 *
 * @since 1.0.0
 */
final class Plugin {

	/**
	 * Addon Version
	 *
	 * @since 1.0.0
	 * @var string The addon version.
	 */
	const VERSION = '1.0.0';

	/**
	 * Minimum Elementor Version
	 *
	 * @since 1.0.0
	 * @var string Minimum Elementor version required to run the addon.
	 */
	const MINIMUM_ELEMENTOR_VERSION = '3.7.0';

	/**
	 * Minimum PHP Version
	 *
	 * @since 1.0.0
	 * @var string Minimum PHP version required to run the addon.
	 */
	const MINIMUM_PHP_VERSION = '7.3';

	/**
	 * Instance
	 *
	 * @since 1.0.0
	 * @access private
	 * @static
	 * @var \artelligence_Toolkit_Addon\Plugin The single instance of the class.
	 */
	private static $_instance = null;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 * @access public
	 * @static
	 * @return \artelligence_Toolkit_Addon\Plugin An instance of the class.
	 */
	public static function instance() {

		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;

	}

	/**
	 * Constructor
	 *
	 * Perform some compatibility checks to make sure basic requirements are meet.
	 * If all compatibility checks pass, initialize the functionality.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function __construct() {

		if ( $this->is_compatible() ) {
			add_action( 'elementor/init', [ $this, 'init' ] );
		}

	}

	/**
	 * Compatibility Checks
	 *
	 * Checks whether the site meets the addon requirement.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function is_compatible() {

		// Check if Elementor installed and activated
		if ( ! did_action( 'elementor/loaded' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_missing_main_plugin' ] );
			return false;
		}

		// Check for required Elementor version
		if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_elementor_version' ] );
			return false;
		}

		// Check for required PHP version
		if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_php_version' ] );
			return false;
		}

		return true;

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have Elementor installed or activated.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function admin_notice_missing_main_plugin() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'artelligence-toolkit-addon' ),
			'<strong>' . esc_html__( 'Artelligence Toolkit Addon', 'artelligence-toolkit-addon' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'artelligence-toolkit-addon' ) . '</strong>'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function admin_notice_minimum_elementor_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'artelligence-toolkit-addon' ),
			'<strong>' . esc_html__( 'Artelligence Toolkit Addon', 'artelligence-toolkit-addon' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'artelligence-toolkit-addon' ) . '</strong>',
			 self::MINIMUM_ELEMENTOR_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required PHP version.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function admin_notice_minimum_php_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'artelligence-toolkit-addon' ),
			'<strong>' . esc_html__( 'Artelligence Toolkit Addon', 'artelligence-toolkit-addon' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'artelligence-toolkit-addon' ) . '</strong>',
			 self::MINIMUM_PHP_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Initialize
	 *
	 * Load the addons functionality only after Elementor is initialized.
	 *
	 * Fired by `elementor/init` action hook.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function init() {

		add_action( 'elementor/widgets/register', [ $this, 'register_widgets' ] );
		add_action( 'elementor/controls/register', [ $this, 'register_controls' ] );
		

	}

	/**
	 * Register Widgets
	 *
	 * Load widgets files and register new Elementor widgets.
	 *
	 * Fired by `elementor/widgets/register` action hook.
	 *
	 * @param \Elementor\Widgets_Manager $widgets_manager Elementor widgets manager.
	 */
	public function register_widgets( $widgets_manager ) {

		require_once( __DIR__ . '/widgets/banner-sliders-widget.php' );
		require_once( __DIR__ . '/widgets/testimonial-widget.php' );
		require_once( __DIR__ . '/widgets/cpt-widget.php' );
		require_once( __DIR__ . '/widgets/artelligence-button-widget.php' );
		require_once( __DIR__ . '/widgets/testimonial-slider-widget-2.php' );
		require_once( __DIR__ . '/widgets/portfolio-post-widget.php' );
		require_once( __DIR__ . '/widgets/services-post-widget.php' );
		require_once( __DIR__ . '/widgets/services-slider-widget.php' );
		require_once( __DIR__ . '/widgets/immersive-button-widget.php' );
		require_once( __DIR__ . '/widgets/our-portfolio-widget.php' );
		require_once( __DIR__ . '/widgets/nav-widget.php' );
		require_once( __DIR__ . '/widgets/artelligence-testimonial-widget.php' );

		$widgets_manager->register( new \Elementor_Banner_Sliders_Widget() );
		$widgets_manager->register( new \Elementor_Testimonial_Widget() );
		$widgets_manager->register( new \Elementor_Cpt_Widget() );
		$widgets_manager->register( new \Elementor_Artelligence_Button_Widget() );
		$widgets_manager->register( new \Elementor_Testimonial_Slider_Widget_2() );
		$widgets_manager->register( new \Elementor_Portfolio_Post_Widget() );
		$widgets_manager->register( new \Elementor_Services_Slider_Widget() );
		$widgets_manager->register( new \Elementor_Services_Post_Widget() );
		$widgets_manager->register( new \Elementor_Immersive_Button_Widget() );
		$widgets_manager->register( new \Elementor_Catering_Service_Widget() );
		$widgets_manager->register( new \Elementor_Artelligence_Testimonial_Widget() );
		$widgets_manager->register( new \Elementor_Nav_Widget() );
	}

	/**
	 * Register Controls
	 *
	 * Load controls files and register new Elementor controls.
	 *
	 * Fired by `elementor/controls/register` action hook.
	 *
	 * @param \Elementor\Controls_Manager $controls_manager Elementor controls manager.
	 */
	public function register_controls( $controls_manager ) {

		require_once( __DIR__ . '/controls/banner-sliders-control.php' );
		require_once( __DIR__ . '/controls/testimonial-control.php' );
		require_once( __DIR__ . '/controls/artelligence-testimonial-control.php' );
		require_once( __DIR__ . '/controls/cpt-control.php' );
		require_once( __DIR__ . '/controls/testimonial-slider-control.php' );
		require_once( __DIR__ . '/controls/artelligence-button-control.php' );
		require_once( __DIR__ . '/controls/portfolio-post-control.php' );
		require_once( __DIR__ . '/controls/services-slider-control.php' );
		require_once( __DIR__ . '/controls/immersive-button-control.php' );
		require_once( __DIR__ . '/controls/services-post-control.php' );
		require_once( __DIR__ . '/controls/nav-control.php' );

		$controls_manager->register( new \Elementor_Banner_Sliders_Control() );
		$controls_manager->register( new \Elementor_Testimonial_Slider_Control() );
		$controls_manager->register( new \Elementor_Testimonial_Control() );
		$controls_manager->register( new \Elementor_Artelligence_Testimonial_Control() );
		$controls_manager->register( new \Elementor_Cpt_Control() );
		$controls_manager->register( new \Elementor_Artelligence_Button_Control() );
		$controls_manager->register( new \Elementor_Portfolio_Post_Control() );
		$controls_manager->register( new \Elementor_Services_Slider_Control() );
		$controls_manager->register( new \Elementor_Immersive_Button_Control() );
		$controls_manager->register( new \Elementor_Services_Post_Control() );
		$controls_manager->register( new \Elementor_Nav_Control() );
	}

}