<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor cpt Widget.
 *
 * Elementor widget that uses the cpt control.
 *
 * @since 1.0.0
 */
class Elementor_Cpt_Widget extends \Elementor\Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve cpt widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'cpt';
	}
	/**
	 * Get widget title.
	 *
	 * Retrieve cpt widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Artelligence Case Studies', 'elementor-cpt-control' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve cpt widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-carousel-loop';
	}

	/**
	 * Register cpt widget controls.
	 *
	 * Add input fields to allow the user to customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {
        
        $this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__( 'Content', 'elementor-cpt-control' ),
				'type' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);
		$this->add_control(
			'number_of_case_studies',
			[
				'label' => __( 'Number of Projects', 'elementor-cpt-control' ),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 6, // Set a default value
				'min' => 1,
				'max' => 21,
				'step' => 1,
			]
		);
		$this->add_control(
			'selected_icon',
			[
				'label' => esc_html__( 'Button Icon', 'elementor' ),
				'type' => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
			]
		);
		$this->add_control(
			'selected_services',
			[
				'label' => esc_html__('Select Services', 'elementor-services-post-control'),
				'type' => \Elementor\Controls_Manager::SELECT2,
				'multiple' => true,
				'options' => $this->get_services_options(), // Define a function to get available services options
			]
		);
		$this->add_control(
			'ai_button_text',
			[
				'label' => esc_html__( 'Button Text', 'elementor-immersive-button-control' ),
				'label_block' => true,
				'placeholder' => __( 'Read More', 'elementor-immersive-button-control' ),
				'type' => 'text',
			]
		);
		$this->add_control(
			'selected_style',
			[
				'label' => esc_html__( 'Select Style', 'elementor-services-post-control' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'style1', // Set the default style
				'options' => [
					'style1' => esc_html__( 'Style 1', 'elementor-services-post-control' ),
					'style2' => esc_html__( 'Style 2', 'elementor-services-post-control' ),
					'style3' => esc_html__( 'Style 3', 'elementor-services-post-control' ),
					'style4' => esc_html__( 'Style 4', 'elementor-services-post-control' ),
					'style5' => esc_html__( 'Style 5', 'elementor-services-post-control' ),
				],
			]
		);
		$this->end_controls_section();
		
		
		$this->start_controls_section(
			'figure_section',
			[
				'label' => esc_html__( 'Figure Dimensions', 'elementor-blog-post-control' ),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);
		
		$this->add_responsive_control(
			'figure_width',
			[
				'label' => esc_html__( 'Figure Width', 'elementor-blog-post-control' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} figure.image' => 'width: {{SIZE}}{{UNIT}};',
				],
			]);
		
		$this->add_responsive_control(
			'figure_height',
			[
				'label' => esc_html__( 'Figure Height', 'elementor-blog-post-control' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} figure.image' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);
		
		$this->end_controls_section();
		$this->start_controls_section(
			'image_section',
			[
				'label' => esc_html__( 'Image Dimensions', 'elementor-blog-post-control' ),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);
		
		$this->add_responsive_control(
			'image_width',
			[
				'label' => esc_html__( 'Image Width', 'elementor-blog-post-control' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} img' => 'width: {{SIZE}}{{UNIT}};',
				],
			]);
		
		$this->add_responsive_control(
			'image_height',
			[
				'label' => esc_html__( 'image Height', 'elementor-blog-post-control' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} img' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'object-fit',
			[
				'label' => esc_html__( 'Object Fit', 'elementor-services-post-control' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'condition' => [
					'image_height[size]!' => '',
				],
				'options' => [
					'' => esc_html__( 'Default', 'elementor-services-post-control' ),
					'fill' => esc_html__( 'Fill', 'elementor-services-post-control' ),
					'cover' => esc_html__( 'Cover', 'elementor-services-post-control' ),
					'contain' => esc_html__( 'Contain', 'elementor-services-post-control' ),
				],
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} img' => 'object-fit: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'object-position',
			[
				'label' => esc_html__( 'Object Position', 'elementor-services-post-control' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'options' => [
					'center center' => esc_html__( 'Center Center', 'elementor-services-post-control' ),
					'center left' => esc_html__( 'Center Left', 'elementor-services-post-control' ),
					'center right' => esc_html__( 'Center Right', 'elementor-services-post-control' ),
					'top center' => esc_html__( 'Top Center', 'elementor-services-post-control' ),
					'top left' => esc_html__( 'Top Left', 'elementor-services-post-control' ),
					'top right' => esc_html__( 'Top Right', 'elementor-services-post-control' ),
					'bottom center' => esc_html__( 'Bottom Center', 'elementor-services-post-control' ),
					'bottom left' => esc_html__( 'Bottom Left', 'elementor-services-post-control' ),
					'bottom right' => esc_html__( 'Bottom Right', 'elementor-services-post-control' ),
				],
				'default' => 'center center',
				'selectors' => [
					'{{WRAPPER}} img' => 'object-position: {{VALUE}};',
				],
				'condition' => [
					'object-fit' => 'cover',
				],
			]
		);
		
		$this->end_controls_section();
		
		$this->start_controls_section(
			'color_section',
			[
				'label' => esc_html__( 'Color', 'elementor-cpt-control' ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'text_color',
			[
				'label'     => esc_html__( 'Text Color', 'elementor-cpt-control' ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .cpt-contents' => 'color: {{VALUE}}',
				],
			]
		);
		$this->add_control(
			'text_color_heading',
			[
				'label'     => esc_html__( 'Text Color 2', 'elementor-cpt-control' ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .study-section .case-box .content span' => 'color: {{VALUE}}',
				],
			]
		);
		$this->add_responsive_control(
			'text_padding',
			[
				'label' => __( ' Text Padding', 'textdomain' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .cpt-contents' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'text_typography_button',
				'label' => __( 'Text Typography', 'textdomain' ),
				'selector' => '{{WRAPPER}} .cpt-contents',
			]
		);
		$this->add_group_control(
			\Elementor\Group_Control_Background::get_type(),
			[
				'name' => 'background1',
				'types' => [ 'classic', 'gradient', 'video' ],
				'selector' => '{{WRAPPER}} .case-box .content',
			]
		);
		$this->add_responsive_control(
			'box_padding',
			[
				'label' => __( 'Content Padding', 'textdomain' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .case-box .content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'content_margin',
			[
				'label' => __( 'Content Margin', 'textdomain' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .case-box .content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_control(
			'content_bg',
			[
				'label'     => esc_html__( 'Content Background', 'elementor-cpt-control' ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .study-section .case-box .content' => 'background-color: {{VALUE}}',
				],
			]
		);
		$this->add_responsive_control(
			'box_radius',
			[
				'label' => esc_html__( 'Box Radius', 'elementor' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .case-box .content' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'desc_margin',
			[
				'label' => __( ' Box Margin', 'textdomain' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .case-box .content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'desc_padding',
			[
				'label' => __( ' Box Padding', 'textdomain' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .case-box .content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_control(
			'title_color',
			[
				'label' => __( 'Title Color', 'elementor-cpt-control' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .case-box h4' => 'color: {{VALUE}}',
				],
			]
		);
		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'title_typography_button',
				'label' => __( 'Title Typography', 'textdomain' ),
				'selector' => '{{WRAPPER}} .case-box h4',
			]
		);

		$this->add_responsive_control(
			'image_radius',
			[
				'label' => esc_html__( 'Image Radius', 'elementor' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .study-section figure img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}} !important;',
				],
			]
		);
		$this->add_responsive_control(
			'image_radius1',
			[
				'label' => esc_html__( 'Figure Radius', 'elementor' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .study-section figure' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}} !important;',
				],
			]
		);
		$this->add_group_control(
			\Elementor\Group_Control_Background::get_type(),
			[
				'name' => 'background_figure',
				'types' => [ 'classic', 'gradient', 'video' ],
				'selector' => '{{WRAPPER}} .study-section figure',
			]
		);
		$this->add_control(
            'button_color',
            [
                'label' => esc_html__('Button Color', 'textdomain'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .btn_wrapper span' => 'color: {{VALUE}}',
                ],
            ]
        );
		$this->add_control(
            'button_color1',
            [
                'label' => esc_html__('Button Background Color', 'textdomain'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .btn_wrapper span' => 'background-color: {{VALUE}}',
                ],
            ]
        );
		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'button_typography_button',
				'label' => __( 'Button Typography', 'textdomain' ),
				'selector' => '{{WRAPPER}} .btn_wrapper span',
			]
		);
        $this->add_responsive_control(
			'button_margin',
			[
				'label' => __( ' Button Margin', 'textdomain' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .btn_wrapper span' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
	
		$this->add_responsive_control(
			'button_padding',
			[
				'label' => __( 'Button Padding', 'textdomain' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .btn_wrapper span' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'button_radius1',
			[
				'label' => esc_html__( 'Button Radius', 'elementor' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .btn_wrapper span' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}} !important;',
				],
			]
		);
		$this->end_controls_section();
		$this->start_controls_section(
			'alignment_section',
			[
				'label' => esc_html__( 'Alignment', 'elementor-cpt-control' ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'alignment',
			[
				'label' => __( 'Alignment', 'elementor-cpt-control' ),
				'type' => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => __( 'Left', 'elementor-cpt-control' ),
						'icon' => 'fa fa-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'elementor-cpt-control' ),
						'icon' => 'fa fa-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'elementor-cpt-control' ),
						'icon' => 'fa fa-align-right',
					],
				],
				'default' => 'left',
				'selectors' => [
					'{{WRAPPER}} .content' => 'text-align: {{VALUE}};',
				],
			]
		);
		$this->add_group_control(
			\Elementor\Group_Control_Background::get_type(),
			[
				'name' => 'background_box',
				'types' => [ 'classic', 'gradient', 'video' ],
				'selector' => '{{WRAPPER}} .study-section .overlay:hover figure',
			]
		);
		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name' => 'border1',
				'label' => __( 'Border', 'textdomain' ),
				'selector' => '{{WRAPPER}} .case-box',
			]
		);
		$this->add_responsive_control(
			'box_radius1',
			[
				'label' => esc_html__( 'Boxes Border Radius', 'elementor' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .case-box' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section();
		$this->start_controls_section(
			'hover_section',
			[
				'label' => esc_html__( 'Box Hover Style', 'elementor-cpt-control' ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);
		
		$this->add_group_control(
			\Elementor\Group_Control_Background::get_type(),
			[
				'name' => 'background_box_hover',
				'types' => [ 'classic', 'gradient', 'video' ],
				'selector' => '{{WRAPPER}} .case-box:hover',
			]
		);
		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name' => 'borderbox',
				'label' => __( 'Border', 'textdomain' ),
				'selector' => '{{WRAPPER}} .case-box:hover',
			]
		);
		$this->add_control(
            'button_color_hover',
            [
                'label' => esc_html__('Button Background Color', 'textdomain'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .case-box:hover .btn_wrapper span' => 'background-color: {{VALUE}}',
                ],
            ]
        );
		$this->add_control(
			'text_color_hover',
			[
				'label'     => esc_html__( 'Text Color', 'elementor-cpt-control' ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .case-box:hover .cpt-contents' => 'color: {{VALUE}}',
				],
			]
		);
		$this->add_control(
			'title_color_hover',
			[
				'label' => __( 'Title Color', 'elementor-cpt-control' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .case-box:hover h4' => 'color: {{VALUE}}',
				],
			]
		);
		$this->add_control(
            'button_hover_color',
            [
                'label' => esc_html__('Button Color', 'textdomain'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .case-box:hover .btn_wrapper span' => 'color: {{VALUE}}',
                ],
            ]
        );
		$this->end_controls_section();
		
		$this->start_controls_section(
			'button_section',
			[
				'label' => esc_html__( 'Button Style', 'elementor-cpt-control' ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
            'icon_color',
            [
                'label' => esc_html__( 'Icon Color', 'textdomain' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .study-section .case-box .circle:before' => 'color: {{VALUE}} !important',
                ],
            ]
        );
		$this->add_control(
            'but_color',
            [
                'label' => esc_html__( 'Button Color', 'textdomain' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .btn_wrapper a' => 'color: {{VALUE}} !important',
                ],
            ]
        );
		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'button_typography1',
				'label' => __( 'Button Typography', 'textdomain' ),
				'selector' => '{{WRAPPER}} .btn_wrapper a',
			]
		);
		$this->add_group_control(
			\Elementor\Group_Control_Background::get_type(),
			[
				'name' => 'background_button1',
				'types' => [ 'classic', 'gradient', 'video' ],
				'selector' => '{{WRAPPER}} .study-section .case-box .circle',
			]
		);
		$this->add_responsive_control(
			'rotate',
			[
				'label' => esc_html__( 'Icon Rotate', 'elementor' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'deg', 'grad', 'rad', 'turn', 'custom' ],
				'default' => [
					'unit' => 'deg',
				],
				'tablet_default' => [
					'unit' => 'deg',
				],
				'mobile_default' => [
					'unit' => 'deg',
				],
				'selectors' => [
					'{{WRAPPER}} i, {{WRAPPER}} .elementor-icon svg' => 'transform: rotate({{SIZE}}{{UNIT}});',
				],
			]
		);
		$this->add_control(
            'icon1_color',
            [
                'label' => esc_html__( 'Icon Hover Color', 'textdomain' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .study-section .case-box:hover .circle::before' => 'color: {{VALUE}} !important',
                ],
            ]
        );
		$this->add_control(
            'but_color_hover',
            [
                'label' => esc_html__( 'Button Hover Color', 'textdomain' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .study-section .case-box:hover .btn_wrapper a' => 'color: {{VALUE}} !important',
                ],
            ]
        );
		$this->add_group_control(
			\Elementor\Group_Control_Background::get_type(),
			[
				'name' => 'background_button2',
				'types' => [ 'classic', 'gradient', 'video' ],
				'selector' => '{{WRAPPER}} .study-section .case-box:hover .circle',
			]
		);
		$this->end_controls_section();
		$this->start_controls_section(
			'width_section',
			[
				'label' => esc_html__('Width', 'elementor-blog-post-control'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);
		
		$this->add_responsive_control(
			'per_box_width',
			[
				'label' => esc_html__( 'Box Width', 'elementor-blog-post-control' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} .box-style' => 'width: {{SIZE}}{{UNIT}};',
				],
			]);
		$this->add_control(
			'box1_padding',
			[
				'label' => __( 'Button Padding', 'textdomain' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .case-box' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_group_control(
			\Elementor\Group_Control_Background::get_type(),
			[
				'name' => 'background_button3',
				'types' => [ 'classic', 'gradient', 'video' ],
				'selector' => '{{WRAPPER}} .case-box',
			]
		);
		$this->end_controls_section();
		
	}
	private function get_services_options() {
		$options = [];
		$args = array(
			'post_type' => 'case_study',
			'posts_per_page' => -1,
			'post_status' => 'publish',
		);
		$services = new WP_Query($args);
	
		if ($services->have_posts()) {
			while ($services->have_posts()) {
				$services->the_post();
				$options[get_the_ID()] = get_the_title();
			}
			wp_reset_postdata();
		}
	
		return $options;
	}
	/**
	 * Render cpt widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		$selected_icon = $settings['selected_icon']['value'];
		$this->add_inline_editing_attributes( 'title_color', 'basic' );
		$this->add_render_attribute( 'title_color', 'class', 'elementor-heading-title' );
		$this->add_render_attribute( 'alignment', 'class', 'your-class' );
		$css = '';
		if ( ! empty( $settings['title_color'] ) ) {
    		$css .= '{{WRAPPER}} h4.elementor-heading-title { color: ' . $settings['title_color'] . '; }';
		}
		if ( $css ) {
    		echo '<style>' . $css . '</style>';
		}
		$number_of_case_studies = $settings['number_of_case_studies'];
		$selected_style = $settings['selected_style'];

        ?>
        <?php
		$selected_services = $settings['selected_services'];
		$args = array(
			'post_type' => 'case_study',
			'orderby' => 'ID',
			'post_status' => 'publish',
			'order' => 'ASC',
			'posts_per_page' => $number_of_case_studies,
			'post__in' => $selected_services, // Include only the selected services
		);

		$result = new WP_Query($args);
		if ($selected_style === 'style1') {
			// Apply Style 1 styles
		echo '<div class="style-1">';
			echo '<div class="study-section">';
				if ( $result-> have_posts() ) {
					?>
					<?php
						echo '<div class="row">';
						$count=0;
					while ( $result->have_posts() ){ 
						
						$result->the_post();
						$url = wp_get_attachment_url( get_post_thumbnail_id(), "thumbnail" );
						$meta = get_post_meta(get_the_ID(),'post_thumbnail', true);
						?>
						<div class="col-lg-4 col-md-4 col-sm-6 col-12">
							<div class="case-box overlay <?php echo esc_attr( $settings['alignment'] ); ?>">
								<figure class="image mb-0">
									<img src="<?php echo $meta ?>" alt="" class="img-fluid">
								</figure>
								<div class="content">
									<span><?php echo get_the_excerpt();?></span>
									<h4 <?php echo $this->get_render_attribute_string( 'title_color' ); ?>><?php the_title(); ?></h4>
									<?php
										if (!empty($settings['selected_icon']['value'])) {
											// User has set values, display them
											?>
											<a href="<?php echo get_post_permalink(); ?>" class="text-decoration-none"><i class="circle <?php echo $settings['selected_icon']['value'] ?>"></i></a>
											<?php
										} else {
											// User hasn't set values, display the default text and icon
											?>
											<a href="<?php echo get_post_permalink(); ?>" class="text-decoration-none"><i class="circle fa-regular fa-angle-right" ></i></a>
											<?php
										} 
									?>
									
								</div>
							</div>
						</div>
						<?php
						$count++;
						if ($count % 3 == 0) {
						echo '</div><div class="row">';
						}
						
					}   
			echo '</div>';
	    }
    
		echo '</div>';
		echo '</div>';
		}
		elseif ($selected_style === 'style2') {
			echo '<div class="style-2">';
		?>
    	<div class="study-section">
		<?php 
			if ( $result-> have_posts() ) {
				?>
				<?php
					echo '<div class="row">';
					$count=0;
				while ( $result->have_posts() ){ 
					
					$result->the_post();
					$url = wp_get_attachment_url( get_post_thumbnail_id(), "thumbnail" );
					$meta = get_post_meta(get_the_ID(),'post_thumbnail', true);
					?>
					<div class="col-xl-3 col-lg-6 col-md-6 col-sm-6 col-12">
						<div class="case-box overlay <?php echo esc_attr( $settings['alignment'] ); ?>">
							<a href="<?php echo get_post_permalink(); ?>" class="text-decoration-none">
								<figure class="image mb-0">
									<img src="<?php echo $meta ?>" alt="" class="img-fluid">
								</figure>
								<div class="content">
									<h4 <?php echo $this->get_render_attribute_string( 'title_color' ); ?>><?php the_title(); ?></h4>
								</div>
							</a>
						</div>
					</div>
					<?php
				}   
				echo '</div>';
			}
		?>
		</div>
		</div>
        <?php
		}
		elseif ($selected_style === 'style3') {
			echo '<div class="style-3">';
		?>
    	<div class="study-section">
		<?php 
			if ( $result-> have_posts() ) {
				?>
				<?php
					$count=0;
				while ( $result->have_posts() ){ 
					
					$result->the_post();
					$url = wp_get_attachment_url( get_post_thumbnail_id(), "thumbnail" );
					$meta = get_post_meta(get_the_ID(),'post_thumbnail', true);
					?>
					<div class="box-style" style="padding:0px 15px;float:left;">
						<div class="case-box <?php echo esc_attr( $settings['alignment'] ); ?>">
							<a href="<?php echo get_post_permalink(); ?>" class="text-decoration-none">
								<figure class="image mb-0">
									<img src="<?php echo wp_get_attachment_url( get_post_thumbnail_id(), "large" ) ?>">
								</figure>
								<div class="content" style="position:relative !important;">
									<h4 <?php echo $this->get_render_attribute_string( 'title_color' ); ?>><?php the_title(); ?></h4>
									<p class="cpt-contents"><?php echo wp_trim_words(get_the_content(), 12) ?></p>
									<div class="btn_wrapper">
										<span><?php echo $settings['ai_button_text'] ?></span>	
									</div>
								</div>
							</a>
						</div>
					</div>
					<?php
				}   
			}
		?>
		</div>
		</div>
        <?php
		}
		elseif ($selected_style === 'style4') {
			echo '<div class="style-4">';
		?>
    	<div class="study-section">
		<?php 
			if ( $result-> have_posts() ) {
				?>
				<?php
					$count=0;
				while ( $result->have_posts() ){ 
					
					$result->the_post();
					$url = wp_get_attachment_url( get_post_thumbnail_id(), "thumbnail" );
					$meta = get_post_meta(get_the_ID(),'post_thumbnail', true);
					?>
					<div class="box-style" style="padding:0px 15px;float:left;">
						<div class="case-box <?php echo esc_attr( $settings['alignment'] ); ?>">
								<div class="row">
									<div class="col-lg-6 col-md-6 col-sm-12">
										<figure class="image mb-0">
											<img src="<?php echo wp_get_attachment_url( get_post_thumbnail_id(), "large" ) ?>">
										</figure>
									</div>
									<div class="col-lg-6 col-md-6 col-sm-12 content-col">
										<div class="content" style="position:relative !important;">
											<h4 <?php echo $this->get_render_attribute_string( 'title_color' ); ?>><?php the_title(); ?></h4>
											<p class="cpt-contents"><?php echo wp_trim_words(get_the_content(), 12) ?></p>
											<div class="btn_wrapper">
											<?php
												if (!empty($settings['selected_icon']['value'])) {
													// User has set values, display them
													?>
													<a href="<?php echo get_post_permalink(); ?>" class="text-decoration-none"><?php echo $settings['ai_button_text'] ?><i class="circle <?php echo $settings['selected_icon']['value'] ?>"></i></a>
													<?php
												} else {
													// User hasn't set values, display the default text and icon
													?>
													<a href="<?php echo get_post_permalink(); ?>" class="text-decoration-none"><?php echo $settings['ai_button_text'] ?><i class="circle fa-regular fa-angle-right" ></i></a>
													<?php
												} 
											?>	
											</div>
										</div>
									</div>
								</div>
						</div>
					</div>
					<?php
				}   
			}
		?>
		</div>
		</div>
        <?php
		}
		elseif ($selected_style === 'style5') {
			echo '<div class="style-4 style-5">';
		?>
    	<div class="study-section">
		<?php 
			if ( $result-> have_posts() ) {
				?>
				<?php
					echo '<div class="row">';
					$count=0;
				while ( $result->have_posts() ){ 
					
					$result->the_post();
					$url = wp_get_attachment_url( get_post_thumbnail_id(), "thumbnail" );
					$meta = get_post_meta(get_the_ID(),'post_thumbnail', true);
					?>
					<div class="col-xl-6 col-lg-6 col-md-6 col-sm-6 col-12">
						<div class="box-style" style="display:flex;">
							<div class="case-box <?php echo esc_attr( $settings['alignment'] ); ?>">
								<a href="<?php echo get_post_permalink(); ?>">
									<figure class="image mb-0">
										<img src="<?php echo wp_get_attachment_url( get_post_thumbnail_id(), "large" ) ?>">
									</figure>
									<div class="content" style="position:relative !important;">
										<h4 <?php echo $this->get_render_attribute_string( 'title_color' ); ?>><?php the_title(); ?></h4>
										<p class="cpt-contents"><?php echo wp_trim_words(get_the_content(), 12) ?></p>
									</div>
								</a>
							</div>
						</div>
					</div>
					<?php
				}
				echo "</div>";   
			}
		?>
		</div>
		</div>
        <?php
		}
	}
	

}
